/*!
 * @file        apm32l0xx_usart.h
 *
 * @brief       This file contains all the functions prototypes for the USART firmware library
 *
 * @version     V1.0.0
 *
 * @date        2021-12-25
 *
 * @attention
 *
 *  Copyright (C) 2020-2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be usefull and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

#ifndef __APM32L0xx_USART_H
#define __APM32L0xx_USART_H

#ifdef __cplusplus
extern "C" {
#endif

#include "apm32l0xx.h"

/** @addtogroup Peripherals_Library Standard Peripheral Library
  @{
*/

/** @addtogroup USART_Driver USART Driver
  @{
*/

/** @addtogroup USART_Enumerations Enumerations
  @{
*/

/**
 * @brief   USART Word Length define
 */
typedef enum
{
    USART_WORD_LEN_8B = 0,
    USART_WORD_LEN_9B = BIT12,
    USART_WORD_LEN_7B = BIT28
} USART_WORD_LEN_T;

/**
 * @brief   USART Parity define
 */
typedef enum
{
    USART_PARITY_NONE   = 0,
    USART_PARITY_EVEN   = BIT10,
    USART_PARITY_ODD    = BIT10 | BIT9
} USART_PARITY_T;

/**
 * @brief   USART mode define
 */
typedef enum
{
    USART_MODE_RX       = BIT2,
    USART_MODE_TX       = BIT3,
    USART_MODE_TX_RX    = BIT2 | BIT3
} USART_MODE_T;

/**
 * @brief   USART Stop bits define
 */
typedef enum
{
    USART_STOP_BIT_1    = 0,
    USART_STOP_BIT_0_5  = BIT12,
    USART_STOP_BIT_2    = BIT13,
    USART_STOP_BIT_1_5  = BIT12 | BIT13
} USART_STOP_BIT_T;

/**
 * @brief   USART hardware flow control define
 */
typedef enum
{
    USART_FLOW_CTRL_NONE     = 0,
    USART_FLOW_CTRL_RTS      = BIT8,
    USART_FLOW_CTRL_CTS      = BIT9,
    USART_FLOW_CTRL_RTS_CTS  = BIT8 | BIT9
} USART_FLOW_CTRL_T;

/**
 * @brief    USART synchronization clock enable/disable
 */
typedef enum
{
    USART_CLKEN_DISABLE,
    USART_CLKEN_ENABLE
} USART_CLKEN_T;

/**
 * @brief    USART Clock polarity define
 */
typedef enum
{
    USART_CLKPOL_LOW,
    USART_CLKPOL_HIGH
} USART_CLKPOL_T;

/**
 * @brief    USART Clock phase define
 */
typedef enum
{
    USART_CLKPHA_1EDGE,
    USART_CLKPHA_2EDGE
} USART_CLKPHA_T;

/**
 * @brief    USART Last bit clock pulse enable
 */
typedef enum
{
    USART_LBCP_DISABLE,
    USART_LBCP_ENABLE
} USART_LBCP_T;

/**
 * @brief    USART inversion Pins
 */
typedef enum
{
    USART_INVERSION_RX,
    USART_INVERSION_TX,
    USART_INVERSION_TX_RX
} USART_INVERSION_T;

/**
 * @brief    USART auto baud rate mode
 */
typedef enum
{
    USART_AUTO_BAUD_RATE_MODE_STARTBIT ,
    USART_AUTO_BAUD_RATE_MODE_FALLINGEDGE,
    USART_AUTO_BAUD_RATE_MODE_FRAME_7F,
    USART_AUTO_BAUD_RATE_MODE_FRAME_55
} USART_AUTO_BAUD_RATE_MODE_T;

/**
 * @brief   Wakeup from Stop mode interrupt flag selection
 */
typedef enum
{
    USART_WAKEUP_SOURCE_ADDRESS = 0x00,
    USART_WAKEUP_SOURCE_START   = 0x02,
    USART_WAKEUP_SOURCE_RXNE    = 0x03
} USART_WAKEUP_SOURCE_T;

/**
 * @brief    USART address mode
 */
typedef enum
{
    USART_ADDRESS_MODE_4B,
    USART_ADDRESS_MODE_7B
} USART_ADDRESS_MODE_T;

/**
 * @brief    USART wakeup method
 */
typedef enum
{
    USART_WAKEUP_IDLE_LINE,
    USART_WAKEUP_ADDRESS_MARK
} USART_WAKEUP_T;

/**
 * @brief    USART LIN break detection length
 */
typedef enum
{
    USART_BREAK_LENGTH_10B,
    USART_BREAK_LENGTH_11B
} USART_BREAK_LENGTH_T;

/**
 * @brief    USART IrDA Low Power
 */
typedef enum
{
    USART_IRDA_MODE_NORMAL,
    USART_IRDA_MODE_LOWPOWER
} USART_IRDA_MODE_T;

/**
 * @brief    USART driver enable polarity select
 */
typedef enum
{
    USART_DE_POL_HIGH,
    USART_DE_POL_LOW
} USART_DE_POL_T;

/**
 * @brief   USART DMA requests
 */
typedef enum
{
    USART_DMA_REQUEST_RX,
    USART_DMA_REQUEST_TX,
    USART_DMA_REQUEST_RX_TX
} USART_DMA_REQUEST_T;

/**
 * @brief   USART request
 */
typedef enum
{
    USART_REQUEST_ABRQ,         //!< Auto Baud Rate Request
    USART_REQUEST_SBQ,          //!< Send Break Request
    USART_REQUEST_MMQ,          //!< Mute Mode Request
    USART_REQUEST_RDFQ,         //!< Receive data flush Request
    USART_REQUEST_TDFQ          //!< Transmit data flush Request
} USART_REQUEST_T;

/**
 * @brief   USART interrupts source
 */
typedef enum
{
    USART_INT_WAKEUP       = BIT22,         //!< Wake up interrupt
    USART_INT_CMIE         = BIT14,         //!< Character match interrupt
    USART_INT_EOBIE        = BIT27,         //!< End of Block interrupt
    USART_INT_RXTOIE       = BIT26,         //!< Receive time out interrupt
    USART_INT_CTSIE        = BIT10,         //!< CTS change interrupt
    USART_INT_LBDIE        = BIT6 ,         //!< LIN Break detection interrupt
    USART_INT_TXBEIE       = BIT7 ,         //!< Tansmit Data Register empty interrupt
    USART_INT_TXCIE        = BIT6 | BIT30,  //!< Transmission complete interrupt
    USART_INT_RXBNEIE      = BIT5 ,         //!< Receive Data buffer not empty interrupt
    USART_INT_IDLEIE       = BIT4 ,         //!< Idle line detection interrupt
    USART_INT_PEIE         = BIT8 ,         //!< Parity Error interrupt
    USART_INT_ERRIE        = BIT0           //!< Error interrupt
} USART_INT_T;

/**
 * @brief   USART Interrupt flag definition
 */
typedef enum
{
    USART_INT_FLAG_WAKEUP  = BIT20,  //!< Wake up flag
    USART_INT_FLAG_CMF     = BIT17,  //!< Character match flag
    USART_INT_FLAG_EOBF    = BIT12,  //!< End of block flag
    USART_INT_FLAG_RXTOF   = BIT11,  //!< Receive time out flag
    USART_INT_FLAG_CTSIF   = BIT9,   //!< CTS interrupt flag
    USART_INT_FLAG_LBDF    = BIT8,   //!< LIN Break detection flag
    USART_INT_FLAG_TXBE    = BIT7,   //!< Transmit data register empty flag
    USART_INT_FLAG_TXC     = BIT6,   //!< Transmission Complete flag
    USART_INT_FLAG_RXBNE   = BIT5,   //!< Receive data buffer not empty flag
    USART_INT_FLAG_IDLE    = BIT4,   //!< Idle Line detection flag
    USART_INT_FLAG_OVRE    = BIT3,   //!< OverRun Error flag
    USART_INT_FLAG_NE      = BIT2,   //!< Noise Error flag
    USART_INT_FLAG_FE      = BIT1,   //!< Framing Error flag
    USART_INT_FLAG_PE      = BIT0    //!< Parity Error flag
} USART_INT_FLAG_T;

/**
 * @brief    USART flag definition
 */
typedef enum
{
    USART_FLAG_RXENACKF    = BIT22,  //!< Receive Enable Acknowledge Flag
    USART_FLAG_TXENACKF    = BIT21,  //!< Transmit Enable Acknowledge Flag
    USART_FLAG_WAKEUP      = BIT20,  //!< Wake Up from stop mode Flag
    USART_FLAG_RWF         = BIT19,  //!< Send Break flag
    USART_FLAG_SBF         = BIT18,  //!< Send Break flag
    USART_FLAG_CMF         = BIT17,  //!< Character match flag
    USART_FLAG_BUSY        = BIT16,  //!< Busy flag
    USART_FLAG_ABRTF       = BIT15,  //!< Auto baud rate flag
    USART_FLAG_ABRTE       = BIT14,  //!< Auto baud rate error flag
    USART_FLAG_EOBF        = BIT12,  //!< End of block flag
    USART_FLAG_RXTOF       = BIT11,  //!< Receive time out flag
    USART_FLAG_CTSF        = BIT10,  //!< CTS Change flag
    USART_FLAG_CTSIF       = BIT9,   //!< CTS interrupt flag
    USART_FLAG_LBDF        = BIT8,   //!< LIN Break Detection Flag
    USART_FLAG_TXBE        = BIT7,   //!< Transmit data register empty flag
    USART_FLAG_TXC         = BIT6,   //!< Transmission Complete flag
    USART_FLAG_RXBNE       = BIT5,   //!< Receive data buffer not empty flag
    USART_FLAG_IDLEF       = BIT4,   //!< Idle Line detection flag
    USART_FLAG_OVRE        = BIT3,   //!< OverRun Error flag
    USART_FLAG_NEF         = BIT2,   //!< Noise Error flag
    USART_FLAG_FEF         = BIT1,   //!< Framing Error flag
    USART_FLAG_PEF         = BIT0    //!< Parity Error flag
} USART_FLAG_T;

/**@} end of group USART_Enumerations*/

/** @addtogroup USART_Structure Data Structure
  @{
*/

/**
 * @brief   USART Config struct definition
 */

typedef struct
{
    uint32_t                    baudRate;           //!< Specifies the baud rate
    USART_WORD_LEN_T            wordLength;         //!< Specifies the word length
    USART_STOP_BIT_T            stopBits;           //!< Specifies the stop bits
    USART_PARITY_T              parity;             //!< Specifies the parity
    USART_MODE_T                mode;               //!< Specifies the mode
    USART_FLOW_CTRL_T           hardwareFlowCtrl;   //!< Specifies the hardware flow control
} USART_Config_T;

/**
 * @brief   USART synchronous communication clock config struct definition
 */
typedef struct
{
    USART_CLKEN_T               clock;          //!< Enable or Disable Clock
    USART_CLKPOL_T              polarity;       //!< Specifies the clock polarity
    USART_CLKPHA_T              phase;          //!< Specifies the clock phase
    USART_LBCP_T                lastBitClock;   //!< Enable or Disable last bit clock
} USART_SyncClockConfig_T;

/**@} end of group USART_Structure*/


/** @addtogroup USART_Fuctions Fuctions
  @{
*/

/** USART peripheral Reset and Configuration */
void USART_Reset(USART_T* usart);
void USART_Config(USART_T* uart, USART_Config_T* configStruct);
void USART_ConfigStructInit(USART_Config_T* configStruct);
void USART_ConfigSyncClock(USART_T* usart, USART_SyncClockConfig_T* syncClockConfig);
void USART_ConfigSyncClockStructInit(USART_SyncClockConfig_T* syncClockConfig);
void USART_Enable(USART_T* usart);
void USART_Disable(USART_T* usart);
void USART_EnableDirectionMode(USART_T* usart, USART_MODE_T mode);
void USART_DisableDirectionMode(USART_T* usart, USART_MODE_T mode);
void USART_ConfigDivider(USART_T* usart, uint8_t divider);
void USART_EnableOverSampling8(USART_T* usart);
void USART_DisableOverSampling8(USART_T* usart);
void USART_EnableMSBFirst(USART_T* usart);
void USART_DisableMSBFirst(USART_T* usart);
void USART_EnableOneBitMethod(USART_T* usart);
void USART_DisableOneBitMethod(USART_T* usart);
void USART_EnableDataInv(USART_T* usart);
void USART_DisableDataInv(USART_T* usart);
void USART_EnableInvPin(USART_T* usart, USART_INVERSION_T invPin);
void USART_DisableInvPin(USART_T* usart, USART_INVERSION_T invPin);
void USART_EnableSWAPPin(USART_T* usart);
void USART_DisableSWAPPin(USART_T* usart);
void USART_EnableReceiverTimeOut(USART_T* usart);
void USART_DisableReceiverTimeOut(USART_T* usart);
void USART_ReceiverTimeOutValue(USART_T* usart, uint32_t timeOut);
void USART_EnableOverrunDetection(USART_T* usart);
void USART_DisableOverrunDetection(USART_T* usart);

/** Auto baud rate detection mode */
void USART_EnableAutoBaudRate(USART_T* usart);
void USART_DisableAutoBaudRate(USART_T* usart);
void USART_ConfigAutoBaudRate(USART_T* usart, USART_AUTO_BAUD_RATE_MODE_T mode);

/** Stop mode */
void USART_EnableStopMode(USART_T* usart);
void USART_DisableStopMode(USART_T* usart);
void USART_ConfigStopModeWakeUpSource(USART_T* usart, USART_WAKEUP_SOURCE_T source);

/** Address */
void USART_Address(USART_T* usart, uint8_t address);
void USART_ConfigAddressDetection(USART_T* usart, USART_ADDRESS_MODE_T address);

/** Transmit and receive */
void USART_TxData(USART_T* usart, uint16_t data);
uint16_t USART_RxData(USART_T* usart);

/*** Mute mode */
void USART_EnableMuteMode(USART_T* usart);
void USART_DisableMuteMode(USART_T* usart);
void USART_ConfigMuteModeWakeUp(USART_T* usart, USART_WAKEUP_T wakeup);

/** LIN mode */
void USART_EnableLINmode(USART_T* usart);
void USART_DisableLINmode(USART_T* usart);
void USART_ConfigLINbreakDetectLength(USART_T* usart, USART_BREAK_LENGTH_T length);

/** Half-duplex mode  */
void USART_EnableHalfDuplex(USART_T* usart);
void USART_DisableHalfDuplex(USART_T* usart);

/** Smartcard mode */
void USART_EnableSmartCard(USART_T* usart);
void USART_DisableSmartCard(USART_T* usart);
void USART_EnableSmartCardNACK(USART_T* usart);
void USART_DisableSmartCardNACK(USART_T* usart);
void USART_ConfigGuardTime(USART_T* usart, uint8_t guardTime);
void USART_ConfigAutoCount(USART_T* usart, uint8_t autoCount);
void USART_ConfigBlockSize(USART_T* usart, uint8_t blockSize);

/** IrDA mode */
void USART_EnableIrDA(USART_T* usart);
void USART_DisableIrDA(USART_T* usart);
void USART_ConfigIrDAMode(USART_T* usart, USART_IRDA_MODE_T IrDAMode);

/** Driver enable Configuration */
void USART_EnableDE(USART_T* usart);
void USART_DisableDE(USART_T* usart);
void USART_ConfigDEPolarity(USART_T* usart, USART_DE_POL_T polarity);
void USART_DEAssertionTimeValue(USART_T* usart, uint8_t value);
void USART_DEDeassertionTimeValue(USART_T* usart, uint8_t value);

/** DMA */
void USART_EnableDMA(USART_T* usart, USART_DMA_REQUEST_T dmaReq);
void USART_DisableDMA(USART_T* usart, USART_DMA_REQUEST_T dmaReq);
void USART_EnableDMAReceptionError(USART_T* usart);
void USART_DisableDMAReceptionError(USART_T* usart);

/** Request */
void USART_EnableRequest(USART_T* usart, USART_REQUEST_T request);

/** Interrupt */
void USART_EnableInterrupt(USART_T* usart, USART_INT_T interrupt);
void USART_DisableInterrupt(USART_T* usart, USART_INT_T interrupt);
uint8_t USART_ReadIntFlag(USART_T* usart, USART_INT_FLAG_T flag);
void USART_ClearIntFlag(USART_T* usart, uint32_t flag);

/** Flag */
uint8_t USART_ReadStatusFlag(USART_T* usart, USART_FLAG_T flag);
void USART_ClearStatusFlag(USART_T* usart, uint32_t flag);

/**@} end of group USART_Fuctions*/
/**@} end of group USART_Driver*/
/**@} end of group Peripherals_Library*/

#ifdef __cplusplus
}
#endif

#endif /* __APM32L0xx_USART_H */
